#!/usr/bin/env python
import os, json, argparse, csv, glob

def find_jsons(runs_dir: str):
    direct = sorted(glob.glob(os.path.join(runs_dir, "*.json")))
    nested = sorted(glob.glob(os.path.join(runs_dir, "runs", "*.json")))
    if nested: return nested
    if direct: return direct
    return sorted(glob.glob(os.path.join(runs_dir, "**", "*.json"), recursive=True))

def get(d, *keys, default=None):
    x = d
    for k in keys:
        if isinstance(x, dict) and k in x: x = x[k]
        else: return default
    return x

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--runs-dir", required=True)
    ap.add_argument("--out", required=True)
    args = ap.parse_args()

    files = find_jsons(args.runs_dir)
    if not files:
        raise SystemExit(f"No JSON files found under {args.runs_dir}")

    rows = []
    for fp in files:
        with open(fp, "r") as f:
            rec = json.load(f)
        row = {
            "gauge": rec.get("gauge"), "L": rec.get("L"), "b": rec.get("b"),
            "kappa": rec.get("kappa"), "f": rec.get("f"), "seed": rec.get("seed"),
            "s_phi":  get(rec, "metrics","radial","s_phi"),
            "r2_phi": get(rec, "metrics","radial","r2_phi"),
            "s_grad": get(rec, "metrics","radial","s_grad"),
            "r2_grad":get(rec, "metrics","radial","r2_grad"),
            "cc_threshold": get(rec,"metrics","cc_meta","threshold_value"),
            "cc_cov":       get(rec,"metrics","cc_meta","mask_coverage"),
            "cc_npixels":   get(rec,"metrics","cc_meta","n_pixels"),
        }
        lens = get(rec, "metrics", "lensing", default={})
        for lk in sorted(lens.keys(), key=lambda z: float(z)):
            row[f"alpha_slope_lambda{lk}"] = get(lens, lk, "alpha_slope")
            row[f"alpha_r2_lambda{lk}"]    = get(lens, lk, "alpha_r2")
        rows.append(row)

    headers = []
    for r in rows:
        for k in r.keys():
            if k not in headers: headers.append(k)

    os.makedirs(os.path.dirname(args.out), exist_ok=True)
    with open(args.out, "w", newline="") as fh:
        w = csv.DictWriter(fh, fieldnames=headers)
        w.writeheader(); w.writerows(rows)
    print(f"Wrote {len(rows)} rows to {args.out}")

if __name__ == "__main__":
    main()
